/**
 * @module UIF (User Interface Focusable)
 * @author iYondaime
 * @version 2.0.0
 *
 * last modified: 2011-06-24 9:28 PM
 */

var UIF = {
    NAV : {
        UP : 1,
        DOWN : 2,
        LEFT : 3,
        RIGHT : 4
    },

    oppositeNav : function (nav) {
        switch (nav) {
        case this.NAV.UP:
            return this.NAV.DOWN;
        case this.NAV.DOWN:
            return this.NAV.UP;
        case this.NAV.LEFT:
            return this.NAV.RIGHT;
        case this.NAV.RIGHT:
            return this.NAV.LEFT;
        default:
            throw new Error("oppositeNav() Invalid nav direction: " + nav);
        }
    },

    isNavLR : function (nav) {
        return nav === this.NAV.LEFT || nav === this.NAV.RIGHT;
    },
    isNavUD : function (nav) {
        return nav === this.NAV.UP || nav === this.NAV.DOWN;
    },

    _plugged : [],

    __listen : function (evt) {
        var move = true,
            nav;
        switch (evt.keyCode) {
        case 38:    // up
            nav = UIF.NAV.UP;
            break;
        case 40:    // down
            nav = UIF.NAV.DOWN;
            break;
        case 37:    // left
            nav = UIF.NAV.LEFT;
            break;
        case 39:    // right
            nav = UIF.NAV.RIGHT;
            break;
        case 13:    // enter
            UIF._plugged[0].enter();
            move = false;
            break;
        default:
            move = false;
        }

        if (move) {
            UIF._plugged[0].navigate(nav);
            evt.preventDefault();   // no spatials
        }
    },

    /**
     * @param UIF.View
     */
    plug : function (uifView) {
        if (this._plugged.length === 0) {
            window.addEventListener('keypress', this.__listen, false);
        }
        else {
            this._plugged[0].blur();
        }
        this._plugged.unshift(uifView);
        uifView.focus();
    },

    unplug : function (uifView) {
        var inx = this._plugged.indexOf(uifView);
        if (inx !== -1) {
            uifView.blur();
            this._plugged.splice(inx, 1);
            if (this._plugged.length === 0) {
                window.removeEventListener('keypress', this.__listen, false);
            }
            else {
                this._plugged[0].focus();
            }
        }
    },

    toString : function () {return "User Interface Focus Manager"}
};




UIF.Mosaic = function (rows, cols, kids) {
    this.init(rows, cols, kids);
};

UIF.Mosaic.prototype = {
    constructor : UIF.Mosaic,

    /**
     * Tries hard to select a kid within: if the end of the row (column)
     * is reached, try selecting first (TODO closest) element in
     * next/prev column (row)
     */
    GREEDY_LR : true,
    GREEDY_UD : true,

    /**
     * If true, then the mosaic remembers the kid from which it was left
     * in given direction. Then, when the mosaic is entered from the opposite
     * direction, that kid gets focused. Otherwise the first kid is selected
     * for down/left and the last for up/right nav
     */
    REMEMBER_LAST : true,

    ROWS : 0,
    COLS : 0,
    currentIndex : -1,
    // _lastSelected : {},
    // kids : [],


    // --- evt handlers
    onfocus : function (el) {},
    onblur : function (el) {},
    onenter : function (el) {},


    /**
     * Set everything to initial state
     * @param {Array | Collection} kids (Sub)Elements. Optional.
     * @param {Number} rows
     * @param {Number} cols
     */
    init : function (rows, cols, kids) {
        this.currentIndex = -1;

        this._lastSelected = this._lastSelected || {};
        for (var dir in UIF.NAV) {
            this._lastSelected[UIF.NAV[dir]] = -1;
        }

        if (arguments.length >= 2) {
            this.set(rows, cols, kids || []);
        }

        return this;
    },

    /**
     * Set rows, cols and - optionally - kids.
     * Everything else remains unchanged. In order to completely
     * clear the mosaic use .init()
     */
    set : function (rows, cols, kids) {
        this.ROWS = rows;
        this.COLS = cols;

        if (kids) {
            this.setKids(kids);
        }
        return this;
    },

    // --- shared (overwrite in prototype only)

    EMPTY_KID : null,
    EMPTY_SELECTION : {el : null /* EMPTY_KID */},

    // assert this.isSelectionEmpty(this.EMPTY_SELECTION)
    isSelectionEmpty : function (selection) {
        return !(selection && selection.el);
    },

    isMosaic : function (object) {
        return object instanceof UIF.Mosaic;
    },

    // --- end of shared

    __areValidKids : function (kids) {
        // https://bugs.opera.com/browse/CORE-37453
        window.__LiveNodeList || (window.__LiveNodeList
            = document.getElementsByTagName('body').constructor);
        window.__StaticNodeList || (window.__StaticNodeList
            = document.querySelectorAll('body').constructor);

        return kids.constructor === Array || kids.constructor === NodeList
            || kids.constructor === HTMLCollection
            || kids.constructor === window.__LiveNodeList
            || kids.constructor === window.__StaticNodeList;
    },

    setKids : function (kids) {
        if (this.__areValidKids(kids)) {
            this.kids = kids;
        }
        else {
            throw new Error("Ivalid kids. Must be Array or NodeList");
        }
    },

    /**
     * Set kid at given index
     * @param {Number} inx
     * @param {Kid} kid
     * @return {Kid} Inserted kid 
     */
    setKid : function (inx, kid) {
        if (inx >= this.COLS * this.ROWS) {
            throw new Error("Out of boundries")
        }

        // fill mid-kids with this.EMPTY_KID
        if (inx > this.kids.length) {
            for (var i = this.kids.length; i < inx; ++i) {
                this.kids[i] = this.EMPTY_KID;
            }
        }

        this.kids[inx] = kid;

        return kid; // allow chaining
    },

    /**
     * Set kid at given row and column
     * @param {Number} r
     * @param {Number} c
     * @param {Kid} kid
     * @return {Kid} Inserted kid 
     */
    setKidAtRowCol : function (r, c, kid) {
        return this.setKid(this.rc2inx(r, c), kid);
    },

    /**
     * Get kid at given index
     * @param {Number} inx
     * @return {Kid}
     */
    kid : function (inx) {
        if (inx >= this.kids.length || inx < 0) {
            return this.EMPTY_KID;   // out of boundries
        }
        else {
            return this.kids[inx];
        }
    },

    /**
     * Get kid at given row, col
     * @param {Number} r
     * @param {Number} c
     * @return {Kid}
     */
    kidAtRowCol : function (r, c) {return this.kid(this.rc2inx(r, c));},

    /** 
     * @param {Number} inx
     * @return {Kid}
     */
    currentKid : function () {return this.kids[this.currentIndex];},

    /**
     * Move selection in given direction.
     * PRE: The mosaic's selected by parent.
     * @param {UIF.NAV} nav Direction to move
     * @return {Selection}
     */
    move : function (nav) {
        // try moving inside the kid
        var selection = this.EMPTY_SELECTION;

        if (this.isMosaic(this.currentKid())) {
            selection = this.currentKid().move(nav);
        }
        if (! this.isSelectionEmpty(selection)) {
            return selection;
        }

        // try moving to another kid within the row/col
        var lastIndex = this.currentIndex;
        selection = this.__moveInLine(nav);

        if (this.isSelectionEmpty(selection)) {
            if (UIF.isNavUD(nav)) {
                if (this.GREEDY_UD) {
                    selection = this.__moveToClosestNeighbourByRows(nav);
                }
            }
            else {
                if (this.GREEDY_LR) {
                    selection = this.__moveToClosestNeighbourByCols(nav);
                }
            }
        }

        if (this.REMEMBER_LAST && this.isSelectionEmpty(selection)) {
            // we're at the edge (moving out)
            this._lastSelected[nav] = lastIndex;
        }

        return selection;
    },

    /**
     * @private
     * Unselects this mosaic and all its selected kids down to the bottom
     * @return {Selection} The just unselected selection =]
     */
    __unselect : function () {
        var kid = this.currentKid(),
            selection = this.isMosaic(kid)
                ? kid.__unselect() : {mosaic : this, el : kid};

        this.currentIndex = -1;

        return selection;
    },

    /**
     * @private
     * Select an Element in a kid of given index moving from given direction.
     *
     * Only if focusable (non-empty) Element was found, this.currentIndex is
     * set to a new value. If the kid is UFI.Mosaic it tries to select its
     * best element {@see UIF.Mosaic.__selectBestElement}, otherwise
     * it selects the kid-element itself.
     * @param {Number} inx
     * @param {UIF.NAV} nav
     * @return {Selection}
     */
    __selectElementInKid : function (inx, nav) {
        var kid = this.kid(inx),
            selection = this.isMosaic(kid)
                ? kid.__selectBestElement(nav)
                : {mosaic : this, el : kid};        

        if (! this.isSelectionEmpty(selection)) {
            this.currentIndex = inx;
        }
        return selection;
    },

    /**
     * @private
     * Select the best element in the mosaic: first, try according to
     * navigation hitsory {@see UIF.Mosaic.__selectElementByNavHistory}.
     * Then, try first element found {@see UIF.Mosaic.__selectFirstElement}
     * @param {UIF.NAV}
     * @return {Selection}
     */
    __selectBestElement : function (nav) {
        var selection = this.__selectElementByNavHistory(nav);
        if (this.isSelectionEmpty(selection)) {
            return this.__selectFirstElement(nav);
        }
        return selection;
    },

    /**
     * @private
     * Going from given direction, select an element that was last selected
     * (and unselected) when moving out from the opposite direction
     * @param {UIF.NAV}
     * @return {Selection}
     */
    __selectElementByNavHistory : function (nav) {
        var inx = this._lastSelected[UIF.oppositeNav(nav)],
            kid = this.kid(inx),
            selection = this.isMosaic(kid)
                ? kid.__selectBestElement(nav)
                : {mosaic : this, el : kid};

        if (! this.isSelectionEmpty(selection)) {
            this.currentIndex = inx;
        }
        return selection;
    },


    /**
     * @private
     * Seek to select first (non-empty) element and select it.
     * @param {Number} beginInx
     * @param {UIF.NAV} nav
     * @return {Selection}
     */
    __selectFirstElement : function (nav) {
        var inx = this.__startIndex(nav),
            selection = this.EMPTY_SELECTION,
            inxMin = 0,
            inxMax = this.kids.length - 1,
            shift = 1;

        if (nav === UIF.NAV.LEFT || nav === UIF.NAV.UP) {
            shift = -1;
        }

        while (
            this.isSelectionEmpty(selection)
            && inx >= inxMin && inx <= inxMax
        ) {
            selection = this.__selectElementInKid(inx, nav);
            inx += shift;
        }
        
        return selection;
    },

    /**
     * @private
     * Move selection to next focusable Element in line (row/column)
     * @param {UIF.NAV} nav
     * @return Selected Element or this.EMPTY_KID
     */
    __moveInLine : function (nav) {

        var inx = this.__movedIndex(nav);

        if (inx === -1) return this.EMPTY_SELECTION;  // out of bounds (no next el.);

        var rc = this.inx2rc(inx),
            selection = this.EMPTY_SELECTION,
            shift, inxMin, inxMax;

        switch (nav) {
        case UIF.NAV.UP:
            shift = -this.COLS;
            // stay in the same column
            inxMin = this.rc2inx(0, rc[1]);
            inxMax = inx;
            break;
        case UIF.NAV.DOWN:
            shift = this.COLS;
            inxMin = inx;
            // stay in the same column
            inxMax = Math.min(
                this.rc2inx(this.ROWS - 1, rc[1]), this.kids.length - 1);                
            break;
        case UIF.NAV.LEFT:
            shift = -1;
            // stay in the same row
            inxMin = this.rc2inx(rc[0], 0);
            inxMax = inx;
            break;
        case UIF.NAV.RIGHT:
            inxMin = inx;
            // stay in the same row
            inxMax = Math.min(
                this.rc2inx(rc[0], this.COLS - 1), this.kids.length - 1);
            shift = 1;
            break;
        default:
            throw new Error("__moveInLine() Invalid nav direction: " + nav);
        }

        while (
            this.isSelectionEmpty(selection)
            && inx >= inxMin && inx <= inxMax
        ) {
            selection = this.__selectElementInKid(inx, nav);
            inx += shift;
        }

        return selection;
    },

    __moveToClosestNeighbourByRows : function (nav) {
        var begining, end, start,
            rc = this.inx2rc(this.currentIndex);

        if (nav === UIF.NAV.UP) {
            begining = 0;
            end = this.rc2inx(rc[0]-1, this.COLS - 1) + 1;
            start = this.rc2inx(rc[0]-1, rc[1]);
        }
        else if (nav === UIF.NAV.DOWN) {
            begining = this.rc2inx(rc[0] + 1, 0);
            end = this.kids.length;
            start = this.rc2inx(rc[0]+1, rc[1]);
        }
        else {
            throw new Error(
                "__moveToClosestNeighbourByRows() Invalid vertical nagivation: "
                + nav
            );
        }

        var self = this,
            selection = this.EMPTY_SELECTION;
        this.__closestNeighbour(this.kids, start, function (kid, inx) {
            selection = self.__selectElementInKid(inx, nav);
            return ! self.isSelectionEmpty(selection);
        }, begining, end);

        return selection;
    },

    __moveToClosestNeighbourByCols : function (nav) {
        var rc = this.inx2rc(this.currentIndex),
            startRow = rc[0],
            currentCol = rc[1],
            minCol = 0, maxCol = this.COLS - 1, colShift;

        if (nav === UIF.NAV.LEFT) {
            // left - up
            colShift = -1;
        }
        else if (nav === UIF.NAV.RIGHT) {
            // right - down
            colShift = 1;
        }
        else {
            throw new Error("__moveToClosestNeighbourByCols()\
 Invalid horizontal nagivation: " + nav);
        }

        
        var c = currentCol + colShift;
        
        var selection = this.EMPTY_SELECTION;
        while (
            this.isSelectionEmpty(selection)
            && c >= minCol && c <= maxCol
        ) {
            selection = this.__moveToClosestNeighbourInCol(
                c, startRow, nav);
            c += colShift;
        }

        return selection;
    },

    __moveToClosestNeighbourInCol : function (col, startRow, nav) {
        var begining = this.rc2inx(0, col),
            end = this.rc2inx(this.ROWS-1, col) + 1,
            step = this.COLS,
            start = this.rc2inx(startRow, col);

        var self = this,
            selection = this.EMPTY_SELECTION;
        this.__closestNeighbour(this.kids, start, function (kid, inx) {
            selection = self.__selectElementInKid(inx, nav);
            return ! self.isSelectionEmpty(selection);
        }, begining, end, step);

        return selection;
    },

    /**
     * @private
     * Find index obtained by shifting the currentIndex in given direction.
     * Do not change currentIndex.
     * @param {UIF.NAV}
     * @return -1 if out of bounds
     */
    __movedIndex : function (nav) {
        var rc = this.inx2rc(this.currentIndex),
            r = rc[0], c = rc[1],
            inx;

        switch (nav) {
        case UIF.NAV.UP:
            r -= 1;
            break;
        case UIF.NAV.DOWN:
            r += 1;
            break;
        case UIF.NAV.LEFT:
            c -= 1;
            break;
        case UIF.NAV.RIGHT:
            c += 1;
            break;
        default:
            throw new Error("__moveIndex() Invalid nav direction: " + nav);
        }

        if (r < 0 || r >= this.ROWS || c < 0 || c >= this.COLS) {
            inx = -1;
        }
        else {
            inx = this.rc2inx(r, c);
            if (inx >= this.kids.length) {
                inx = -1;
            }
        }

        return inx;
    },

    /**
     * @private
     * Find start index for movement in given direction.
     * Do not change currentIndex.
     * @param {UIF.NAV}
     * @param {Number} -1 if there's nothing to select
     */
    __startIndex : function (nav) {
        var r = 0, c = 0;
        if (nav === UIF.NAV.LEFT || nav === UIF.NAV.UP) {
            r = this.ROWS - 1;
            c = this.COLS - 1;
        }

        var inx = this.rc2inx(r, c);

        if (inx >= this.kids.length) {
            inx = this.kids.length - 1;
        }

        return inx;
    },

    /**
     * @private
     * {@see UIF.View.selectElement}
     * @param {Number} inx
     * @return {Kid} Selected kid
     */
    __selectKid : function (inx) {
        this.currentIndex = inx;
        return this.kid(inx);
    },

    /**
     * @private
     * {@see UIF.View.selectElement}
     * @param {Number} row
     * @param {Number} col
     * @return {Kid} Selected kid
     */
    __selectKidAtRowCol : function (r, c) {
        return this.__selectKid(this.rc2inx(r, c));
    },

    // utils

    inx2rc : function (index) {
        var row = index / this.COLS >> 0,
            col = index % this.COLS;
        return [row, col];
    },

    rc2inx : function (r, c) {
        return this.COLS * r + c;
        //return (this.COLS === window.Infinity ? 0 : this.COLS * r) + c;
    },

    /**
     * Find the index closest to startInx, where the element
     * fulfills given callback condition. Does not check the index itself.
     * @param {Array | Collection | List} arr
     * @param {Number} startInx
     * @param {Function} callback Called with array's item and checked index
     * @param {Number} begining Start index (included), defaults to 0;
     * @param {Number} end End index (excluded, defaults to arr's length)
     * @param {Number} step Index's shift. Optional. Defaults to 1;
     */
    __closestNeighbour : function (arr, startInx, callback, begining, end, step) {
        var begining = begining || 0,
            end = end === undefined ? arr.length : end;

        if (end <= begining) return -1;

        var step = step || 1;
            rightInx = startInx + step,
            leftInx = startInx - step,
            endReached = rightInx >= end,
            beginingReached = leftInx < begining;

        while (! endReached || ! beginingReached) {
            if (! beginingReached) {
                if (callback(arr[leftInx], leftInx)) return leftInx;
                leftInx -= step;
                beginingReached = leftInx < begining;
            }
            if (! endReached) {
                if (callback(arr[rightInx], rightInx)) return rightInx;
                rightInx += step;
                endReached = rightInx >= end;
            }
        }

        return -1;
    },

    toString : function () {return "[object UIF.Mosaic]";}
};



/**
 * @class
 */
UIF.View = function (rows, cols, kids) {
    if (arguments.length >= 2) {
        this.init(rows, cols, kids);
    }
};
UIF.View.prototype = new UIF.Mosaic();


UIF.View.ONEDGE = {
    STOP : 1,
    NEXT : 2,
    // NEXT2STOP : 2,
    // NEXT2LOOP : 3,
    LOOP : 4
};


(function () {
    this.constructor = UIF.View;

    this.super = UIF.Mosaic.prototype;

    this.ONEDGE_LR = UIF.View.ONEDGE.STOP;
    this.ONEDGE_UD = UIF.View.ONEDGE.STOP;

    /**
     * @readonly
     */
    //this.selection = this.EMPTY_SELECTION;

    this.init = function (rows, cols, kids) {
        this.super.init.call(this, rows, cols, kids);

        this.selection = this.EMPTY_SELECTION;

        return this;
    };

    /**
     * Select element defined by coordinates given in arguments.
     * If the view is active the selection's onfocus is called.
     *
     * Starting from this view, a kid is selected, then the
     * kid-mosaic's kid and so on. If last-found kid is a mosaic, it's
     * first element is selected: 
     * >> this.selectElement(0) the first focusable element there is.
     *
     * If the path points to non-element (or an empty one) this.selection
     * is blurred and set to empty:
     * >> this.selectElement(-1) blurs the view completely
     *
     * @param {[Number | Array]+} If the argument is a number its
     * used as index; if it's an array it's interpreted as [row, column].
     */
    this.selectElement = function (/* path */) {
        this.__selectElement(arguments, UIF.NAV.DOWN);
    };

    /**
     * @private
     * Select given element by path and nav (the latter used only if
     * the path ends at a mosaic)
     * @param {Array} path Items or the array are either indexes {Number} or row-col
     * coordinates {Array[Number]}. See the arguments to .selectElement()
     * @param {UIF.NAV} nav 
     */
    this.__selectElement = function (path, nav) {
        var undoneSelection = this.__unselect(),
            element2select,
            kid = this,
            mosaic = this,
            index;
 
        for (var i = 0; i < path.length; ++i) {
            index = path[i];
            if (index instanceof Array) {
                kid = kid.__selectKidAtRowCol(index[0], index[1])
            }
            else {
                kid = kid.__selectKid(index);
            }

            if (this.isMosaic(kid)) {
                mosaic = kid;
            }
            else break;
        }

        selection = this.isMosaic(kid)
            ? kid.__selectBestElement(nav)
            : {mosaic : mosaic, el : kid};

        // if the DOM was manipulated it's possible that
        // undoneSelection.el !== this.selection.el
        // in that case just blur them both
        if (undoneSelection.el !== this.selection.el) {
            this.__blurSelection(selection);
            this.blur();
        }

        this.__focusSelection(selection);
    };

    // helpers
    this.plug = function () {UIF.plug(this);};
    this.unplug = function () {UIF.unplug(this);};


    // UIF PROTOCOL

    /**
     * Move selection (focusize next element if non-empty)
     * @param {UIF.NAV}
     */
    this.navigate = function (nav) {
        var selection = this.isUnselected() ?
            this.__selectBestElement(nav) : this.move(nav);

        if (this.isSelectionEmpty(selection)) {
            // we're at the edge...
            selection = this.__onedge(nav);
        }

        if (! this.isSelectionEmpty(selection)) {
            this.__focusSelection(selection);
        }
    };

    /**
     * Enter selected element (if non-empty)
     */
    this.enter = function () {
        if (! this.isSelectionEmpty(this.selection)) {
            this.selection.mosaic.onenter(this.selection.el);
        }
    };

    /**
     * Blur (call onblur) this.selection
     * (but do not set this.selection to empty)
     */
    this.blur = function () {
        this.__blurSelection(this.selection);
    };

    /**
     * Focus (call onfocus) this.selection iff
     * the selection is nonempty and the view is active
     */
    this.focus = function () {
        this.__focusSelection(this.selection);
    };

    // PRIVATE

    /**
     * @private
     * Focus selection (which also means setting it as this.selection)
     * selection's onfocus is called iff the selection is non-empty
     * and the view is active
     */
    this.__focusSelection = function (selection) {
        this.blur();
        this.selection = selection;
        if (! this.isSelectionEmpty(selection) && this.__isActive()) {
            selection.mosaic.onfocus(selection.el);
        }
    };

    /**
     * @private
     * Blur (call onblur) on given selection iff it is non-empty
     */
    this.__blurSelection = function (selection) {
        if (! this.isSelectionEmpty(selection)) {
            selection.mosaic.onblur(selection.el);
        }
    };

    /**
     * @private
     * Is the UIF.View currently in focus (first among plugged)
     */
    this.__isActive = function () {
        return UIF._plugged[0] === this;
    };

    /**
     *
     */
    this.isUnselected = function () {
        return this.currentIndex < 0 || this.currentIndex >= this.kids.length;
    };

    this.__onedge = function (nav) {
        if (UIF.isNavLR(nav)) {
            switch (this.ONEDGE_LR) {
            case UIF.View.ONEDGE.LOOP:
                // stay in row
                var rc = this.inx2rc(this.currentIndex);
                rc[1] = nav === UIF.NAV.RIGHT ? 0 : this.COLS - 1;

                this.__selectElement([rc], nav);
                if (this.isSelectionEmpty(this.selection)) {
                    return this.__moveInLine(nav);
                }
                else {
                    // has been focused in .selectElement();
                    // no need to focus it again
                    return this.EMPTY_KID;
                }
            case UIF.View.ONEDGE.NEXT:
                return this.__moveByRows(nav);
            default:
                return this.EMPTY_SELECTION;
            }
        }
        else if (UIF.isNavUD(nav)) {
            switch (this.ONEDGE_UD) {
            case UIF.View.ONEDGE.LOOP:
                // stay in col
                
                var rc = this.inx2rc(this.currentIndex);
                rc[0] = nav === UIF.NAV.DOWN ? 0 : this.ROWS - 1;

                this.__selectElement([rc], nav);
                if (this.isSelectionEmpty(this.selection)) {
                    return this.__moveInLine(nav);
                }
                else {
                    return this.EMPTY_SELECTION;  // see comment above
                }
            case UIF.View.ONEDGE.NEXT:
                return this.__moveByCols(nav);
            default:
                return this.EMPTY_SELECTION;
            }
        }
    };

    /**
     * Move selection by rows. Depending the nav argument go left-up or
     * right-down, starting from last-in-row item above (or first-in-row item 
     * below, respectively)
     * @param {UIF.NAV} nav
     */
    this.__moveByRows = function (nav) {
        var shift, startCol, currentRow = this.inx2rc(this.currentIndex)[0],
            selection = this.EMPTY_SELECTION,
            inxMin = 0,
            inxMax = this.kids.length;

        if (nav === UIF.NAV.RIGHT) {
            // down - right
            shift = 1;
            startCol = 0;
        }
        else if (nav === UIF.NAV.LEFT) {
            // up - left
            shift = -1;
            startCol = this.COLS - 1;
        }
        else {
            throw new Error("__moveByRows() Invalid LR nagivation: " + nav);
        }

        var inx = this.rc2inx(currentRow + shift, startCol);

        while (this.isSelectionEmpty(selection) && inx >= inxMin && inx <= inxMax) {
            selection = this.__selectElementInKid(inx, nav);
            inx += shift;
        }

        return selection;
    };

    /**
     * Move selection by cols. Depending on nav argument go down-right or
     * up-left, starting from first-in-col item on the right (or last-in-col
     * item on the left, respectively)
     * @param {UIF.NAV} nav
     */
    this.__moveByCols = function (nav) {
        var currentColumn = this.inx2rc(this.currentIndex)[1], startRow,
            rowShift, colShift,
            minRow = 0, maxRow = this.ROWS -1,
            minCol = 0, maxCol = this.COLS - 1,
            selection = this.EMPTY_SELECTION;

        if (nav === UIF.NAV.UP) {
            // left - up
            colShift = -1;
            rowShift = -1;
        }
        else if (nav === UIF.NAV.DOWN) {
            // right - down
            colShift = 1;
            rowShift = 1
        }
        else {
            throw new Error("__moveByCols() Invalid UD nagivation: " + nav);
        }

        startRow = rowShift < 0 ? maxRow : minRow;

        var inx, r,
            c = currentColumn + colShift;

        while (this.isSelectionEmpty(selection) && c >= minCol && c <= maxCol) {
            r = startRow;
            while (this.isSelectionEmpty(selection) && r >= minRow && r <= maxRow) {
                inx = this.rc2inx(r, c);
                selection = this.__selectElementInKid(inx, nav);
                r += rowShift;
            }
            c += colShift;
        }

        return selection;
    };

    this.toString = function () {return "[object UIF.View]"};
}).call(UIF.View.prototype);
